
/* eslint-disable no-shadow */

'use strict';

var collections = require('*/cartridge/scripts/util/collections');
var Site = require('dw/system/Site');
var redsysConstants = require('*/cartridge/scripts/constants/redsysConstants');
var Logger = require('dw/system/Logger');
var logger = Logger.getLogger('Redsys', 'redsys');
// eslint-disable-next-line no-use-before-define
const redsysCreditCardPreferences = getRedsysCreditCardPreferences();
// eslint-disable-next-line no-use-before-define
const redsysBizumPreferences = getRedsysBizumPreferences();
// eslint-disable-next-line no-use-before-define
var redsysSession = parseRedsysSessionData();

/**
 * Parse Redsys session data
 * @returns {Object|null} session data | null
 */
function parseRedsysSessionData() {
    // eslint-disable-next-line no-undef
    var redsysSession = session.privacy.redsys;
    if (redsysSession) {
        return JSON.parse(redsysSession);
    }

    return null;
}

/**
 * Multiply number by 100 and return it.
 *
 * @param {number} amount Amount
 * @returns {number} Multiplied number by 100
 */
function calculateAmount(amount) {
    return Math.round(amount * 100);
}

/**
 * Checks if payment with Redsys - Credit card is enabled
 * @returns {boolean} Is payment enabled
 */
function isRedsysCreditCardEnabled() {
    var PaymentMgr = require('dw/order/PaymentMgr');
    var activePaymentMethods = PaymentMgr.getActivePaymentMethods();
    var redsysCreditCard = redsysConstants.REDSYS_CREDIT_CARD;
    var isEnabled = false;

    // eslint-disable-next-line consistent-return
    Array.some(activePaymentMethods, function (paymentMethod) {
        if (paymentMethod.paymentProcessor.ID === redsysCreditCard) {
            isEnabled = true;
            return true;
        }
    });

    return isEnabled;
}

/**
 * Checks if payment with Redsys - Bizum is enabled
 * @returns {boolean} Is payment enabled
 */
function isRedsysBizumEnabled() {
    var PaymentMgr = require('dw/order/PaymentMgr');
    var activePaymentMethods = PaymentMgr.getActivePaymentMethods();
    var redsysBizum = 'REDSYS_BIZUM';
    var isEnabled = false;

    // eslint-disable-next-line consistent-return
    Array.some(activePaymentMethods, function (paymentMethod) {
        if (paymentMethod.paymentProcessor.ID === redsysBizum) {
            isEnabled = true;
            return true;
        }
    });

    return isEnabled;
}

/**
 * Get Redsys operation mode
 * @returns {string} Operation mode
 */
function getRedsysMode() {
    var site = Site.current;
    return site.getCustomPreferenceValue('redsys_operation_mode').value;
}

/**
 * Get Redsys redirection url
 * @returns {string} Redirection url
 */
function getRedirectionUrl() {
    if (getRedsysMode() === redsysConstants.REDSYS_MODE_LIVE) {
        return redsysConstants.REDSYS_REDIRECTION_URL_LIVE;
    }

    return redsysConstants.REDSYS_REDIRECTION_URL_TEST;
}

/**
 * Repeat character.
 * @param {string} char - character
 * @param {int} length - length
 * @returns {string} character / space
 */
function repeatCharacter(char, length) {
    if (length > 0) {
        return char.repeat(length);
    }

    return '';
}

/**
 * Get SFRA 5.3.0 compatibility mode status
 * @returns {boolean} true if SFRA 5.3.0 compatibility is enabled
 */
function redsysSFRA5CompatibilityModeEnabled() {
    var site = Site.current;
    return site.getCustomPreferenceValue('redsys_sfra5_compatibility');
}

/**
 * Get Bizum custom preferences
 * @returns {Object} custom preferences
 */
function getRedsysBizumPreferences() {
    var site = Site.current;
    var redsysBizumPreferences = {
        merchantCode: site.getCustomPreferenceValue('redsys_bizum_merchant_code'),
        merchantSecret: site.getCustomPreferenceValue('redsys_bizum_merchant_secret'),
        merchantTerminal: site.getCustomPreferenceValue('redsys_bizum_merchant_terminal'),
        limitTotal: site.getCustomPreferenceValue('redsys_bizum_limit_total')
    };

    return redsysBizumPreferences;
}

/**
 * Get Redsys preferences
 * @returns {Object} custom preferences
 */

function getRedsysCreditCardPreferences() {
    var site = Site.current;

    var redsysCreditCardPreferences = {
        merchantCode: site.getCustomPreferenceValue('redsys_credit_card_merchant_code'),
        merchantSecret: site.getCustomPreferenceValue('redsys_credit_card_merchant_secret'),
        merchantTerminal: site.getCustomPreferenceValue('redsys_credit_card_merchant_terminal'),
        integrationMode: site.getCustomPreferenceValue('redsys_credit_card_integration_mode').value
    };

    return redsysCreditCardPreferences;
}


/**
 * Get Required configuration values to pay with Redsys using Bizum
 * @param {string} orderID - Customer's order ID
 * @param {string} amount - Amount to pay
 * @returns {Object} - Required config to pay with Bizum
 */
function getBizumConfig(orderID, amount) {
    var URLUtils = require('dw/web/URLUtils');

    var obj = {};
    // eslint-disable-next-line no-undef
    const currencyCode = session.currency.currencyCode;
    if (currencyCode !== 'EUR') {
        obj = {
            error: true,
            message: 'currency error'
        };

        logger.error('Error in getBizumConfig to order {1}', orderID);
        return obj;
    }

    obj = {
        DS_MERCHANT_AMOUNT: calculateAmount(amount),
        DS_MERCHANT_CURRENCY: '978',
        DS_MERCHANT_MERCHANTCODE: redsysBizumPreferences.merchantCode,
        DS_MERCHANT_MERCHANTURL: URLUtils.https('Redsys-Notify').toString(),
        DS_MERCHANT_ORDER: orderID,
        DS_MERCHANT_TERMINAL: redsysBizumPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '0',
        DS_MERCHANT_URLKO: URLUtils.https('Redsys-KO', 'orderID', orderID).toString(),
        DS_MERCHANT_URLOK: URLUtils.https('Redsys-OK', 'orderID', orderID).toString(),
        DS_MERCHANT_PAYMETHODS: 'z'
    };

    return obj;
}

/**
 * Get merchant identifier if customer has saved payments
 * @param {string} paymentUUID - Payment UUID
 * @returns {Object} object - ds_merchant_identifier
 */
function getMerchantIdentifier(paymentUUID) {
    var merchantIdentifier = null;
    // eslint-disable-next-line no-undef
    if (customer.authenticated && customer.registered) {
        // eslint-disable-next-line no-undef
        var paymentInstruments = customer.getProfile().getWallet().getPaymentInstruments(redsysConstants.REDSYS_CREDIT_CARD);
        var storedPaymentUUID = paymentUUID != null ? paymentUUID : redsysSession.storedPaymentUUID;
        collections.forEach(paymentInstruments, function (paymentInstrument) {
            if (paymentInstrument.UUID === storedPaymentUUID) {
                merchantIdentifier = paymentInstrument.creditCardToken;
                return;
            }
        });
    }

    return merchantIdentifier;
}

/**
 * Check if customer has saved payment instruments
 * @returns {boolean} Return true if customer has saved payment instruments
 */
function hasSavedPaymentInstruments() {
    var hasSavedPaymentInstruments = false;
        // eslint-disable-next-line no-undef
    var customer = session.customer;
    if (customer.authenticated) {
        var paymentInstruments = customer.getProfile().getWallet().getPaymentInstruments(redsysConstants.REDSYS_CREDIT_CARD);
        hasSavedPaymentInstruments = paymentInstruments.length > 0;
    }

    return hasSavedPaymentInstruments;
}

/**
 * Get Required configuration values to pay with Redsys using credit card with redirection mode
 * @param {string} orderID - Customer's order ID
 * @param {string} amount - Amount to pay
 * @returns {Object} - Required config to pay with credit card
 */
function getRedsysCreditCardRedirectionConfig(orderID, amount) {
    var URLUtils = require('dw/web/URLUtils');
    var currencies = require('*/cartridge/config/currencies');

    var obj = {};
    // eslint-disable-next-line no-undef
    const currencyCode = session.currency.currencyCode;
    var currencyObj = !currencyCode
    ? currencies[0]
    : currencies.filter(function (currency) {
        return currency.currencyCode === currencyCode;
    })[0];

    if (!currencyObj) {
        obj = {
            error: true,
            message: 'currency error'
        };

        logger.error('Error in getRedsysCreditCardRedirectionConfig params to order: {0}', orderID);
        return obj;
    }

    var saveCreditCard = redsysSession.saveCard ? redsysConstants.REDSYS_MERCHANT_IDENTIFIER_REQUIRED : '';

    obj = {
        DS_MERCHANT_AMOUNT: calculateAmount(amount),
        DS_MERCHANT_CURRENCY: currencyObj.currencyNumber,
        DS_MERCHANT_MERCHANTCODE: redsysCreditCardPreferences.merchantCode,
        DS_MERCHANT_MERCHANTURL: URLUtils.https('Redsys-Notify').toString(),
        DS_MERCHANT_ORDER: orderID,
        DS_MERCHANT_TERMINAL: redsysCreditCardPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '0',
        DS_MERCHANT_URLKO: URLUtils.https('Redsys-KO', 'orderID', orderID).toString(),
        DS_MERCHANT_URLOK: URLUtils.https('Redsys-OK', 'orderID', orderID).toString(),
        DS_MERCHANT_IDENTIFIER: saveCreditCard
    };

    if (hasSavedPaymentInstruments()) {
        var merchantIdentifier = getMerchantIdentifier();
        if (merchantIdentifier) {
            obj.DS_MERCHANT_IDENTIFIER = merchantIdentifier;
        }
    }

    return obj;
}

/**
 * Get Required configuration values call iniciaPeticionREST endpoint
 * @param {string} orderID - Customer's order ID
 * @param {string} amount - Amount to pay
 * @returns {Object} - Required config to pay with credit card
 */
function getRedsysIniciaPeticionConfig(orderID, amount) {
    var currencies = require('*/cartridge/config/currencies');

    var obj = {};
    // eslint-disable-next-line no-undef
    const currencyCode = session.currency.currencyCode;
    var currencyObj = !currencyCode
    ? currencies[0]
    : currencies.filter(function (currency) {
        return currency.currencyCode === currencyCode;
    })[0];

    if (!currencyObj || !redsysSession) {
        obj = {
            error: true,
            message: 'currency error'
        };

        logger.error('Error in getRedsysIniciaPeticionConfig params to order: {0}', orderID);
        return obj;
    }

    obj = {
        DS_MERCHANT_AMOUNT: amount,
        DS_MERCHANT_CURRENCY: currencyObj.currencyNumber,
        DS_MERCHANT_EMV3DS: {
            threeDSInfo: 'CardData'
        },
        DS_MERCHANT_MERCHANTCODE: redsysCreditCardPreferences.merchantCode,
        DS_MERCHANT_ORDER: orderID,
        DS_MERCHANT_TERMINAL: redsysCreditCardPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '0'
    };

    var merchantIdentifier = getMerchantIdentifier();
    if (merchantIdentifier) {
        obj.DS_MERCHANT_IDENTIFIER = merchantIdentifier;
    } else {
        obj.DS_MERCHANT_IDOPER = redsysSession.creditCardToken;
    }
    var tmp = redsysSession;
    return obj;
}

/**
 * Get Required configuration values call iniciaPeticionREST endpoint
 * @param {string} orderID - Customer's order ID
 * @param {string} amount - Amount to pay
 * @returns {Object} - Required config to pay with credit card
 */
function getRedsysIniciaPeticionRefundConfig(order, amount) {
    var currencies = require('*/cartridge/config/currencies');

    var obj = {};
    // eslint-disable-next-line no-undef
    const currencyCode = order.getCurrencyCode();
    var currencyObj = !currencyCode
    ? currencies[0]
    : currencies.filter(function (currency) {
        return currency.currencyCode === currencyCode;
    })[0];

    if (!currencyObj) {
        obj = {
            error: true,
            message: 'currency error'
        };

        logger.error('Error in getRedsysIniciaPeticionConfig params to order: {0}', order.orderNo);
        return obj;
    }

    obj = {
        DS_MERCHANT_AMOUNT: "00000000" + amount,
        DS_MERCHANT_CURRENCY: currencyObj.currencyNumber,
        DS_MERCHANT_MERCHANTCODE: redsysCreditCardPreferences.merchantCode,
        DS_MERCHANT_ORDER: order.orderNo,
        DS_MERCHANT_TERMINAL: redsysCreditCardPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '3'
    };
    return obj;
}

/**
 * Get Required configuration values to call trataPeticionREST endpoint
 * @param {Object} iniciaPeticionResponse - iniciaPeticion response object
 * @param {string} orderID - Customer's order ID
 * @param {string} amount - Amount to pay
 * @returns {Object} - Required config to pay with credit card
 */
function getRedsysTrataPeticionConfig(iniciaPeticionResponse, orderID, amount) {
    var currencies = require('*/cartridge/config/currencies');

    var obj = {};
    var DsMerchantEMV3DS = null;
    var threeDSMethodURL = iniciaPeticionResponse.Ds_EMV3DS.threeDSMethodURL;
    // eslint-disable-next-line no-undef
    const currencyCode = session.currency.currencyCode;
    var currencyObj = !currencyCode
    ? currencies[0]
    : currencies.filter(function (currency) {
        return currency.currencyCode === currencyCode;
    })[0];

    if (!currencyObj || !redsysSession) {
        obj = {
            error: true,
            message: 'Params error'
        };

        logger.error('Error in getRedsysTrataPeticionConfig params to order: {0}', orderID);
        return obj;
    }

    var saveCreditCard = redsysSession.saveCard ? redsysConstants.REDSYS_MERCHANT_IDENTIFIER_REQUIRED : '';

    obj = {
        DS_MERCHANT_AMOUNT: amount,
        DS_MERCHANT_CURRENCY: currencyObj.currencyNumber,
        DS_MERCHANT_MERCHANTCODE: redsysCreditCardPreferences.merchantCode,
        DS_MERCHANT_ORDER: orderID,
        DS_MERCHANT_TERMINAL: redsysCreditCardPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '0',
        DS_MERCHANT_IDENTIFIER: saveCreditCard
    };

    if (iniciaPeticionResponse.Ds_EMV3DS.protocolVersion === redsysConstants.REDSYS_EMV3DS_PROTOCOL_VERSION_N03DS_V2) {
        DsMerchantEMV3DS = {
            threeDSInfo: redsysConstants.REDSYS_EMV3DS_AUTHENTICACIONDATA,
            protocolVersion: redsysConstants.REDSYS_EMV3DS_PROTOCOL_VERSION_102,
            browserAcceptHeader: redsysConstants.REDSYS_EMV3DS_BROWSER_ACCEPT_HEADER,
            browserUserAgent: redsysSession.userAgent
        };
    } else {
        DsMerchantEMV3DS = {
            threeDSInfo: redsysConstants.REDSYS_EMV3DS_AUTHENTICACIONDATA,
            protocolVersion: iniciaPeticionResponse.Ds_EMV3DS.protocolVersion,
            browserAcceptHeader: redsysConstants.REDSYS_EMV3DS_BROWSER_ACCEPT_HEADER,
            browserUserAgent: redsysSession.userAgent,
            browserJavaEnabled: redsysSession.javaEnabled,
            browserJavascriptEnabled: redsysSession.javaScriptEnabled,
            browserLanguage: redsysSession.browserLanguage,
            browserColorDepth: redsysSession.browserColorDepth,
            browserScreenHeight: redsysSession.browserScreenHeight,
            browserScreenWidth: redsysSession.browserScreenHeight,
            browserTZ: redsysSession.browserTZ,
            threeDSServerTransID: iniciaPeticionResponse.Ds_EMV3DS.threeDSServerTransID,
            notificationURL: redsysConstants.REDSYS_INSITE_NOTIFICATION_URL + '?order=' + orderID,
            threeDSCompInd: threeDSMethodURL ? 'Y' : 'N'
        };
    }

    obj.DS_MERCHANT_EMV3DS = DsMerchantEMV3DS;

    var merchantIdentifier = getMerchantIdentifier();
    if (merchantIdentifier) {
        obj.DS_MERCHANT_IDENTIFIER = merchantIdentifier;
    } else {
        obj.DS_MERCHANT_IDOPER = redsysSession.creditCardToken;
    }

    return obj;
}

/**
 * Get Required configuration values to call trataPeticionREST endpoint with authentication
 * @param {string} orderID - Customer's order ID
 * @param {string} authData - Authentication data
 * @returns {Object} - Required config to authenticate
 */
function getRedsysAutenticacionConfig(orderID, authData) {
    var currencies = require('*/cartridge/config/currencies');
    var customObjectMgr = require('dw/object/CustomObjectMgr');

    var obj = {};
    // eslint-disable-next-line no-undef
    var currencyCode = session.currency.currencyCode;
    var DS_MERCHANT_EMV3DS = null;
    var currencyObj = !currencyCode
    ? currencies[0]
    : currencies.filter(function (currency) {
        return currency.currencyCode === currencyCode;
    })[0];

    var redsysCustomObj = customObjectMgr.getCustomObject('RedsysNotifications', orderID);

    if (!currencyObj) {
        obj = {
            error: true,
            message: 'Params error'
        };

        logger.error('Error in getRedsysAutenticacionConfig params to order: {0}', orderID);
        return obj;
    }

    obj = {
        DS_MERCHANT_AMOUNT: redsysCustomObj.custom.amount,
        DS_MERCHANT_CURRENCY: currencyObj.currencyNumber,
        DS_MERCHANT_MERCHANTCODE: redsysCreditCardPreferences.merchantCode,
        DS_MERCHANT_ORDER: orderID,
        DS_MERCHANT_TERMINAL: redsysCreditCardPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '0'
    };

    if (redsysCustomObj.custom.protocolVersion === redsysConstants.REDSYS_EMV3DS_PROTOCOL_VERSION_102 || !authData.cres) {
        DS_MERCHANT_EMV3DS = {
            threeDSInfo: redsysConstants.REDSYS_EMV3DS_CHALLENGE_REQUEST_RESPONSE,
            protocolVersion: authData.cRes ?
                redsysCustomObj.custom.protocolVersion :
                redsysConstants.REDSYS_EMV3DS_PROTOCOL_VERSION_102,
            PARes: authData.PaRes,
            MD: authData.MD
        };
    } else {
        DS_MERCHANT_EMV3DS = {
            threeDSInfo: redsysConstants.REDSYS_EMV3DS_CHALLENGE_REQUEST_RESPONSE,
            protocolVersion: redsysCustomObj.custom.protocolVersion,
            cres: authData.cres
        };
    }

    obj.DS_MERCHANT_EMV3DS = DS_MERCHANT_EMV3DS;

    var merchantIdentifier = getMerchantIdentifier(redsysCustomObj.custom.storedPaymentUUID);
    if (merchantIdentifier) {
        obj.DS_MERCHANT_IDENTIFIER = merchantIdentifier;
    } else {
        obj.DS_MERCHANT_IDOPER = redsysCustomObj.custom.creditCardToken;
    }

    return obj;
}

/**
 * Get Required configuration values to call trataPeticionREST endpoint with authentication
 * @param {string} UUID - UUID of saved payment instrument
 * @param {string} token - Token of saved payment instrument
 * @param {string} orderID - Random number
 * @returns {Object} - Required config to authenticate
 */
function getRedsysBorraPreferenciaConfig(UUID, token, orderID) {
    var obj = {};
    var merchantIdentifier = getMerchantIdentifier(UUID);

    obj = {
        DS_MERCHANT_MERCHANTCODE: redsysCreditCardPreferences.merchantCode,
        Ds_Merchant_Identifier: merchantIdentifier,
        DS_MERCHANT_ORDER: orderID,
        DS_MERCHANT_TERMINAL: redsysCreditCardPreferences.merchantTerminal,
        DS_MERCHANT_TRANSACTIONTYPE: '44'
    };

    return obj;
}

/**
 * Encrypt message using 3DES encryption
 * @param {string} message - message to be encrypted, usually orderId is sent
 * @param {Object} key - key to be used on the 3DES encryption, this is a key provided by Redsys
 * @returns {string} Encrypted code
 */
function encrypt3DES(message, key) {
    var Bytes = require('dw/util/Bytes');
    var Cipher = require('dw/crypto/WeakCipher');
    var Encoding = require('dw/crypto/Encoding');

    // We calculate the next multiple of 8 of the message length
    var length = Math.ceil(message.length / 8) * 8;
    // We extend the message to the following multiple of 8 with null characters
    var messageExtended = message + repeatCharacter('\0', length - message.length);
    // We use 8 bytes of null characters
    var iv = new Bytes('\0\0\0\0\0\0\0\0');
    // We create a new byte and encode to base64
    var ivEncoded64 = Encoding.toBase64(iv); // AAAAAAAAAAA=
    // We encrypt it using 3DES (Triple DES) Encryption
    var CipherObj = new Cipher();
    var encrypted = CipherObj.encrypt(messageExtended, key, 'DESede/CBC/NOPADDING', ivEncoded64, 1);
    // We decode the 3DES encrypted value, because SFCC returns a base64 encoded value
    var encryptedDecoded = Encoding.fromBase64(encrypted);

    return encryptedDecoded;
}

/**
 * Dedode Redsys API base64 encoded responses
 * @param {string} response - Key to be used on the 3DES encryption, this is a key provided by Redsys
 * @returns {string} Encoded data
 */
function decodeResponse(response) {
    var StringUtils = require('dw/util/StringUtils');
    var parsedResponse = JSON.parse(response.object);

    if (parsedResponse.errorCode) {
        return {
            error: true,
            message: parsedResponse.errorCode
        };
    }

    return JSON.parse(StringUtils.decodeBase64(parsedResponse.Ds_MerchantParameters));
}

/**
 * Create merchant signature
 * @param {string} key - Key to be used on the 3DES encryption, this is a key provided by Redsys
 * @param {Object} data - All the purchase data such as order price, order id...
 * @param {Object} order - Customer's order ID
 * @returns {Object} Redsys API response object
 */
function createMerchantSignatureRequest(key, data, order) {
    var Bytes = require('dw/util/Bytes');
    var Encoding = require('dw/crypto/Encoding');
    var MAC = require('dw/crypto/Mac');

    // We generate Ds_MerchantParameters
    var dataJSON = JSON.stringify(data);
    var merchantParameters = Encoding.toBase64(new Bytes(dataJSON));

    // We diversify order ID with the key
    var diversifiedKey = encrypt3DES(order, key);
    // MAC256 of Ds_MerchantParameters together with the key
    var MAC256 = new MAC(MAC.HMAC_SHA_256);
    var res = MAC256.digest(merchantParameters, diversifiedKey);
    // Code data to base64
    var encodedRes = Encoding.toBase64(res);

    return encodedRes;
}

/**
 * Make payment parameters
 * @param {string} action - REST method
 * @param {string} paymentMethod - Payment method
 * @param {Object} paramsInput - Paramns needed
 * @param {*} orderId - Customer's order ID
 * @returns {Object} parameters
 */
function makeParameters(action, paymentMethod, paramsInput, orderId) {
    var StringUtils = require('dw/util/StringUtils');
    var merchantSecret;
    var params = paramsInput;
    var parameters = {};

    parameters.action = action;

    if (paymentMethod === redsysConstants.REDSYS_CREDIT_CARD) {
        merchantSecret = redsysCreditCardPreferences.merchantSecret;
    }
    if (paymentMethod === redsysConstants.REDSYS_BIZUM) {
        merchantSecret = redsysBizumPreferences.merchantSecret;
    }

    var DsSignature = createMerchantSignatureRequest(merchantSecret, params, orderId);
    parameters.payload = {
        Ds_SignatureVersion: redsysConstants.RESDSYS_DS_SIGNATURE_VERSION,
        Ds_MerchantParameters: StringUtils.encodeBase64(JSON.stringify(params)),
        Ds_Signature: DsSignature
    };

    return parameters;
}

module.exports = {
    parseRedsysSessionData: parseRedsysSessionData,
    isRedsysCreditCardEnabled: isRedsysCreditCardEnabled,
    redsysSFRA5CompatibilityModeEnabled: redsysSFRA5CompatibilityModeEnabled,
    getRedsysBizumPreferences: getRedsysBizumPreferences,
    getRedsysCreditCardPreferences: getRedsysCreditCardPreferences,
    getBizumConfig: getBizumConfig,
    getRedsysCreditCardRedirectionConfig: getRedsysCreditCardRedirectionConfig,
    getRedsysIniciaPeticionConfig: getRedsysIniciaPeticionConfig,
    getRedsysTrataPeticionConfig: getRedsysTrataPeticionConfig,
    getRedsysAutenticacionConfig: getRedsysAutenticacionConfig,
    getRedsysBorraPreferenciaConfig: getRedsysBorraPreferenciaConfig,
    isRedsysBizumEnabled: isRedsysBizumEnabled,
    getRedsysMode: getRedsysMode,
    getRedirectionUrl: getRedirectionUrl,
    makeParameters: makeParameters,
    decodeResponse: decodeResponse,
    calculateAmount: calculateAmount,
    hasSavedPaymentInstruments: hasSavedPaymentInstruments,
    getRedsysIniciaPeticionRefundConfig: getRedsysIniciaPeticionRefundConfig,
};
