package es.redsys.rest.api.model.message;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.json.JsonArray;
import javax.json.JsonNumber;
import javax.json.JsonObject;
import javax.json.JsonString;
import javax.json.JsonValue;
import javax.json.JsonValue.ValueType;

import es.redsys.rest.api.constants.RestConstants;
import es.redsys.rest.api.constants.RestConstants.TransactionType;
import es.redsys.rest.api.model.RestRequest;
import es.redsys.rest.api.utils.RestTransformUtils;
import es.redsys.rest.api.utils.RestUtils;

public class RestOperationMessage implements RestRequest {
	
	//Params
	
	/** Merchant code (FUC) */
	private String merchant;
	
	/** Terminal code */
	private String terminal;
	
	/** Operation order code */
	private String order;
	
	/** Operation ID code */
	private String operID;
	
	/** Operation type */
	private TransactionType transactionType;
	
	/** Currency code (ISO 4217)
	 * @see https://en.wikipedia.org/wiki/ISO_4217 */
	private String currency;
	
	/** Operation amount, withot decimal separation */
	private String amount;
	
	/** Card Number */
	private String cardNumber;
	
	/** Card ExpiryDate */
	private String cardExpiryDate;
	
	/** Card CVV2 */
	private String cvv2;
	
	/** Other operation parameters */
	private Map<String, String> parameters;

	/** 3DSecure information */
	private Map<String, String> emv;
	
	/** DCC information */
	private Map<String, String> dcc;
	
	
	//Getters and Setters
	
	/** gets the amount of the operation
	 * @return the operation amount
	 */
	public String getAmount() {
		return amount;
	}

	/** sets de amount of the operation
	 * @param amount without decimal separation
	 */
	public void setAmount(String amount) {
		this.amount = amount;
	}

	/** get currency code
	 * @return the currency code (numeric ISO_4217)
	 */
	public String getCurrency() {
		return currency;
	}

	/** sets the currency code
	 * @param currency the currency code (numeric ISO_4217 )
	 */
	public void setCurrency(String currency) {
		this.currency = currency;
	}

	/** gets the merchant code (FUC)
	 * @return the merchant code
	 */
	public String getMerchant() {
		return merchant;
	}

	/** sets the merchant code
	 * @param merchant merchant code
	 */
	public void setMerchant(String merchant) {
		this.merchant = merchant;
	}

	/** gets the operation ID
	 * @return the operation ID
	 */
	public String getOperID() {
		return operID;
	}

	/** sets the operation ID
	 * @param operID the operation ID
	 */
	public void setOperID(String operID) {
		this.operID = operID;
	}

	/** gets the operation order code (max length 12)
	 * @return the operation order (max length 12)
	 */
	public String getOrder() {
		return order;
	}

	/** sets the operation order (max length 12)
	 * @param order (max length 12)
	 */
	public void setOrder(String order) {
		this.order = order;
	}

	/** gets the terminal code
	 * @return the terminal code
	 */
	public String getTerminal() {
		return terminal;
	}

	/** sets the terminal code
	 * @param terminal terminal code (max lenght 3)
	 */
	public void setTerminal(String terminal) {
		this.terminal = terminal;
	}

	/** gets the operation type 
	 * @return the operation type
	 */
	public TransactionType getTransactionType() {
		return transactionType;
	}

	/** sets the operation type
	 * @param transactionType the operation type
	 */
	public void setTransactionType(TransactionType transactionType) {
		this.transactionType = transactionType;
	}
	
	/**
	 * @return the cardNumber
	 */
	public String getCardNumber() {
		return cardNumber;
	}

	/**
	 * @param cardNumber the cardNumber to set
	 */
	public void setCardNumber(String cardNumber) {
		this.cardNumber = cardNumber;
	}

	/**
	 * @return the cardExpiryDate
	 */
	public String getCardExpiryDate() {
		return cardExpiryDate;
	}

	/**
	 * @param cardExpiryDate the cardExpiryDate to set
	 */
	public void setCardExpiryDate(String cardExpiryDate) {
		this.cardExpiryDate = cardExpiryDate;
	}

	/**
	 * @return the cvv2
	 */
	public String getCvv2() {
		return cvv2;
	}

	/**
	 * @param cvv2 the cvv2 to set
	 */
	public void setCvv2(String cvv2) {
		this.cvv2 = cvv2;
	}

	/** gets the 3DSecure Information
	 * @return 3DSecure Information
	 */
	public Map<String, String> getEmv() {
		return emv;
	}

	/** sets the 3DSecure Request Info
	 *@param 3DSecure param
	 *@param 3DSecure value
	 */
	public void setEmv(Map<String, String> emv) {
		this.emv = emv;
	}
	
	/** sets the 3DSecure Request Info
	 *@param 3DSecure param and value
	 */
	public void setEmv(String emv) {
		this.emv = RestTransformUtils.JSONToMap(emv);
	}
	
	
	
	//Methods
	
	/** Flag for reference creation (card token for merchant to use in other operations) */
	public void createReference() {
		addParameter(RestConstants.REQUEST_MERCHANT_IDENTIFIER, RestConstants.REQUEST_MERCHANT_IDENTIFIER_REQUIRED);
	}
	
	/** Method for using a reference created before for the operation
	 * @param reference the reference string to be used
	 */
	public void useReference(String reference) {
		addParameter(RestConstants.REQUEST_MERCHANT_IDENTIFIER, reference);
	}
	
	/** Flag for direct payment operation. Direct payment operation implies:
	 * 1) No-secure operation	 
	 * 2) No-DCC operative appliance 
	 */
	public void useDirectPayment() {
		addParameter(RestConstants.REQUEST_MERCHANT_DIRECTPAYMENT, RestConstants.REQUEST_MERCHANT_DIRECTPAYMENT_TRUE);
	}
	
	/** 
	 * For use a MOTO Payment
	 */
	public void useMOTOPayment() {
		addParameter(RestConstants.REQUEST_MERCHANT_DIRECTPAYMENT, RestConstants.REQUEST_MERCHANT_DIRECTPAYMENT_MOTO);
	}
	
	
	/** Other optional operation parameters 
	 * @param key key parameter
	 * @param value key value 
	 */
	public void addParameter(String key, String value) {
		if (null == parameters) {
			parameters = new HashMap<String, String>();
		}
		parameters.put(key, value);
	}
	
	/** returns every parameter
	 * @return the parameter map
	 */
	public Map<String, String> getParameters() {
		return this.parameters;
	}


	/** Other optional operation parameters 
	 * @param key key parameter
	 * @param value key value 
	 */
	public void addEmvParameter(String key, String value) {
		if (null == emv) {
			emv = new HashMap<String, String>();
		}
		emv.put(key, value);
	}
	
	
	/** DCC parameters 
	 * @param key key parameter
	 * @param value key value 
	 */
	public void addDCCParameter(String key, String value) {
		if (null == dcc) {
			dcc = new HashMap<String, String>();
		}
		dcc.put(key, value);
	}
	
	/** 
	 * Method for the first COF operation
	 */	
	public void setCOFOperation(String cofType) {
		
		addParameter(RestConstants.REQUEST_MERCHANT_COF_INI, RestConstants.REQUEST_MERCHANT_COF_INI_TRUE);
		addParameter(RestConstants.REQUEST_MERCHANT_COF_TYPE, cofType);
	}
	
	/** 
	 * Method for a COF operation
	 */	
	public void setCOFTxnid(String txnid) {
		addParameter("DS_MERCHANT_COF_TXNID", txnid);
	}
	
	public Map<String, String> getDcc() {
		return dcc;
	}

	public void setDcc(Map<String, String> dcc) {
		this.dcc = dcc;
	}

	/** 
	 * method for a DCC operation 
	 */
	public void dccOperation(String monedaDCC, String importeDCC) {
		
		
		
//		String jsonDCCString = "{\"monedaDCC\":" +monedaDCC + ",\"importeDCC\":" + importeDCC + "\"}\" ";
//		System.out.println("El string que se envía es: " + jsonDCCString);

		addDCCParameter("monedaDCC", monedaDCC);
		addDCCParameter("importeDCC", importeDCC);
	}
	
	/** 
	 * Method for set the EMV3DS protocolVersionV1 parameters 
	 */	
	public void setEMV3DSParamsV1() {
		
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_THREEDSINFO, RestConstants.REQUEST_MERCHANT_EMV3DS_AUTHENTICACIONDATA);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_PROTOCOLVERSION, RestConstants.REQUEST_MERCHANT_EMV3DS_PROTOCOLVERSION_102);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_ACCEPT_HEADER, RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_ACCEPT_HEADER_VALUE);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_USER_AGENT, RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_USER_AGENT_VALUE);
	}
	
	/**
	 *  Method for set the EMV3DS protocolVersionV2 parameters 
	 * @param protocolVersion 3EMV3DS authentications version
	 * @param browserAcceptHeader 3DSMethod return value
	 * @param browserUserAgent 3DSMethod return value
	 * @param browserJavaEnable 3DSMethod return value
	 * @param browserJavaScriptEnabled 3DSMethod return value
	 * @param browserLanguage 3DSMethod return value
	 * @param browserColorDepth 3DSMethod return value
	 * @param browserScreenHeight 3DSMethod return value
	 * @param browserScreenWidth 3DSMethod return value
	 * @param browserTZ 3DSMethod return value
	 * @param threeDSServerTransID 3DSMethod return value
	 * @param notificationURL Authentication URL
	 * @param threeDSCompInd 3DSMethod return value
	 */
	public void setEMV3DSParamsV2(String protocolVersion, String browserAcceptHeader, String browserUserAgent, String browserJavaEnable, 
			String browserJavaScriptEnabled,String browserLanguage, String browserColorDepth, String browserScreenHeight,String browserScreenWidth,
			String browserTZ, String threeDSServerTransID, String notificationURL, String threeDSCompInd) {
		
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_THREEDSINFO, RestConstants.REQUEST_MERCHANT_EMV3DS_AUTHENTICACIONDATA);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_PROTOCOLVERSION, protocolVersion);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_ACCEPT_HEADER, browserAcceptHeader);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_USER_AGENT, browserUserAgent);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_JAVA_ENABLE, browserJavaEnable);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_JAVASCRIPT_ENABLE, browserJavaEnable);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_LANGUAGE, browserLanguage);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_COLORDEPTH, browserColorDepth );
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_SCREEN_HEIGHT, browserScreenHeight );
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_SCREEN_WIDTH,browserScreenWidth);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_BROWSER_TZ, browserTZ );
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_THREEDSSERVERTRANSID, threeDSServerTransID );
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_NOTIFICATIONURL, notificationURL);
		addEmvParameter(RestConstants.REQUEST_MERCHANT_EMV3DS_THREEDSCOMPIND, threeDSCompInd);
	}
	
	
	/** 
	 * Method for set the authentication exemption for V2 EMV3DS
	 * @param exemption: constant of the exemption the commerce want to use
	 */	
	public void setExemption(String exemption) {
		if ("LWV".equals(exemption)) {
			addParameter(RestConstants.REQUEST_MERCHANT_EXEMPTION, RestConstants.REQUEST_MERCHANT_EXEMPTION_VALUE_LWV);
		}else if("TRA".equals(exemption)) {
			addParameter(RestConstants.REQUEST_MERCHANT_EXEMPTION, RestConstants.REQUEST_MERCHANT_EXEMPTION_VALUE_TRA);
		}else if("MIT".equals(exemption)) {
			addParameter(RestConstants.REQUEST_MERCHANT_EXEMPTION, RestConstants.REQUEST_MERCHANT_EXEMPTION_VALUE_MIT);
		}else if("COR".equals(exemption)) {
			addParameter(RestConstants.REQUEST_MERCHANT_EXEMPTION, RestConstants.REQUEST_MERCHANT_EXEMPTION_VALUE_COR);
		}else if("ATD".equals(exemption)) {
			addParameter(RestConstants.REQUEST_MERCHANT_EXEMPTION, RestConstants.REQUEST_MERCHANT_EXEMPTION_VALUE_ATD);
		}else if("NDF".equals(exemption)) {
			addParameter(RestConstants.REQUEST_MERCHANT_EXEMPTION, RestConstants.REQUEST_MERCHANT_EXEMPTION_VALUE_NDF);
		}		
	}
	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer("");
		sb.append("InsiteOperationMessage [");
			if (null != parameters) {
				for (Map.Entry<String, String> entry : parameters.entrySet()) {
				    String key = entry.getKey();
				    String value = entry.getValue();
				    sb.append(key + "=" + value + ", ");
				}
			}
			sb.append("merchant=" + merchant + ", ");
			sb.append("terminal=" + terminal + ", ");
			sb.append("order=" + order + ", ");
			sb.append("operID=" + operID + ", ");
			sb.append("transactionType=" + transactionType + ", ");
			sb.append("currency=" + currency + ", ");
			sb.append("amount=" + amount + ", ");
			if (emv != null)
				sb.append("emv=" + RestTransformUtils.MapToJSON(emv));
		sb.append("]");
		return sb.toString();
	}
}